/**************** (c) 2006 Florent COSTE **********************
     
PROJECT  : ST7MC Rx battery charger
COMPILER : ST7 COSMIC

MODULE  :  regul.c
LIBRARY VERSION  :  1.0

CREATION DATE :    01.2006
AUTHOR :      Florent COSTE	

-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

DESCRIPTION :   routine for closed LOOP operation
              
-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

 ******************************************************************************
 THE SOFTWARE INCLUDED IN THIS FILE IS FOR GUIDANCE ONLY. THE AUTHOR 
 SHALL NOT BE HELD LIABLE FOR ANY DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES
 WITH RESPECT TO ANY CLAIMS ARISING FROM USE OF THIS SOFTWARE.
 ******************************************************************************

******************************************************************************/  

#include "lib.h"
#include "mtc.h"
#include "ST7MC_hr.h"
#include "it_ST7MC.h"
#include "ports.h"
#include "adc.h"
#include "regul.h"


#define Error_slip_MAX   (s16)2048 
#define Error_slip_MIN   (s16)-2048  

#define MTIM_MAX_FREQ    MTC_CLOCK/8     // 16Mhz/8 -> 2Mhz

#define Ki_Min 	20
#define Kp_Min 	30
//#define Ki_Min 	10
//#define Kp_Min 	15
#define Ki_Max 	100
#define Kp_Max 	180
//#define Ki_Max 	70
//#define Kp_Max 	120

//u8 Ki=20;
//u8 Kp=30;
u8 Ki=Ki_Min;
u8 Kp=Kp_Min;


//--------------------------------------
// Variables
//--------------------------------------

static s32 VoltageIntegralTerm;
static BOOL MinPiOut;
BOOL MaxPiOut;
u16 Current_Charge;//, Current_Discharge; 
u16 Duty_fan_control;

/*-------------------Main program-------------------- */


/*Initialisation of Integral term of PI*/
void Init_PI(void)
{
VoltageIntegralTerm = 0;
}


/*-----------------------------------------------------------------------------
ROUTINE Name :  regul_PI

Description:	Compute PI output (0 (PI min) to 1023 (PI max)) according to Ki,
				Kp, sampling time, and target electrical frequency.
                
Input/Output:	u16/u16 
Comments: 	None
-----------------------------------------------------------------------------*/
u16 regul_PI(u16 Target_Current,u16 Target_Voltage)      // return 10 bits value
{      
s32 Voltage_slip_s32,DeltaVoltage_slip_s32,Newpi_32;
s16 NewPIoutput, Error_slip,Error;
u16 output,current;
s32 temp;

/********************** Compute PI output ***************************************/

if (Target_Current != 0)		// current regulation
	{
	Error = (s16)(Target_Current - Get_Current());  

	if (Error > (s16)(Error_slip_MAX)) 
		{
		Error_slip = Error_slip_MAX;
		}
	else if (Error < (s16)(Error_slip_MIN))
		{
		Error_slip = Error_slip_MIN;
		}
	else Error_slip = (s16)Error;
	}
else		// voltage regulation
	{
	Error = (s16)(Get_Battery_Voltage() - Target_Voltage); 
	if (State == CHARGE_LIPO_PHASE_2)	Error = -Error;
	
	if (Error > (s16)(Error_slip_MAX)) 
		{
		Error_slip = Error_slip_MAX;
		}
	else if (Error < (s16)(Error_slip_MIN))
		{
		Error_slip = Error_slip_MIN;
		}
	else Error_slip = (s16)Error;
	Error_slip = Error_slip/4;
	}


/********************** Compute Ki & Kp terms ********************************/

if ( (Target_Current != 0) && (Error_slip <= 0) )  // only if current > target current
	{
	if (Error_slip <= (s16)(-1000))  
				VoltageIntegralTerm = 0;// more than 1.0A difference? cancel integral term!
	temp = 64;
	temp *= Error_slip;
	temp *= (s16)((s16)Ki_Max-(s16)Ki_Min);
//	temp /= (s32)(64*Error_slip_MAX); // div by 2048*64
	temp /= (s32)(64*(s32)Error_slip_MAX); // div by 2048*64
	temp = -temp;
	temp += (s16)Ki_Min;
	Ki = (u8)temp;

	temp = 64;
	temp *= Error_slip;
	temp *= (s16)((s16)Kp_Max-(s16)Kp_Min);
//	temp /= (s32)(64*Error_slip_MAX); // div by 2048*64
	temp /= (s32)(64*(s32)Error_slip_MAX); // div by 2048*64
	temp = -temp;
	temp += (s16)Kp_Min;
	Kp = (u8)temp;
	}
else 
	{
//	Ki = Ki_Min;		// current < target current or voltage regulation
//	Kp = Kp_Min;
	Ki = 20;		// current < target current or voltage regulation
	Kp = 30;
	}
	
/********************** Compute Proportional term ********************************/

Voltage_slip_s32 = Kp * (s32)Error_slip; 

/********************** Compute Integral term ************************************/
// If modulation is maximum, integral term must be "frozen" 

DeltaVoltage_slip_s32 = ( Ki * SAMPLING_TIME * (s32)Error_slip)/256; 
 
if( ((Error_slip>0) && !MaxPiOut) || ((Error_slip<0) && !MinPiOut) )
	{
	if ((VoltageIntegralTerm >= 0) && (DeltaVoltage_slip_s32 >= 0))
		{
		if (( (u32)VoltageIntegralTerm + (u32)DeltaVoltage_slip_s32 ) > S32_MAX)
			VoltageIntegralTerm = S32_MAX;	// Avoid IntTerm Overflow
		else VoltageIntegralTerm += DeltaVoltage_slip_s32; // "integral" output
		}
	else if ((VoltageIntegralTerm < 0) && (DeltaVoltage_slip_s32 < 0))
		{
		if (( (u32)VoltageIntegralTerm + (u32)DeltaVoltage_slip_s32 ) <= S32_MAX)
			VoltageIntegralTerm = S32_MIN;	// Avoid IntTerm Overflow
		else VoltageIntegralTerm += DeltaVoltage_slip_s32; // "integral" output
		}
	 else
		VoltageIntegralTerm += DeltaVoltage_slip_s32; // "integral" output 
	}

    if ((VoltageIntegralTerm >= 0) && (Voltage_slip_s32 >= 0))
		{
		if (( (u32)VoltageIntegralTerm + (u32)Voltage_slip_s32 ) > S32_MAX)
			Newpi_32 = S32_MAX;	// Avoid IntTerm Overflow
		else Newpi_32 = (VoltageIntegralTerm + Voltage_slip_s32); //  output
		}
	else if ((VoltageIntegralTerm < 0) && (Voltage_slip_s32 < 0))
		{
		if (( (u32)VoltageIntegralTerm + (u32)Voltage_slip_s32 ) <= S32_MAX)
			Newpi_32 = S32_MIN;	// Avoid IntTerm Overflow
		else Newpi_32 = (VoltageIntegralTerm + Voltage_slip_s32); //  output
		}
	 else

		Newpi_32 = (VoltageIntegralTerm + Voltage_slip_s32); //  output 
	
    NewPIoutput = (s16)( Newpi_32 /256);                                    

if ( NewPIoutput < 0 )
	{
	output = 0;
	MinPiOut = TRUE;
	}
else if ( NewPIoutput > 1000 )
	{
	output = 1000;
	MaxPiOut = TRUE;  // Set ClampFlag if modulation reaches maximum value 
	}
else
	{
	output = NewPIoutput;
	MinPiOut = FALSE;
	MaxPiOut = FALSE;
	}

return (output);  // return PI output
}
 
 
 
/*-----------------------------------------------------------------------------
ROUTINE Name :  regul_Fan

Description:	Set Fan Speed set according to temperature
                
Input/Output:	u16/u16 
Comments: 	None
-----------------------------------------------------------------------------*/
void regul_Fan(u16 temperature)      
{      
u32 tp;

if (temperature <= _35deg) Duty_fan_control = 0;
else if (temperature >= _65deg) Duty_fan_control = 5000;
else
{
tp = (u32) ((u16)temperature - (u16)_35deg) * 5000;
tp /= (u8)((u16)_65deg - (u16)_35deg);
Duty_fan_control = (u16) tp;
if (Duty_fan_control <= 500) Duty_fan_control = 500;
else if (Duty_fan_control >= 4500) Duty_fan_control = 5000;
} 
}

/*** (c) 2006 ****************** END OF FILE ***/


