/**************** (c) 2006 Florent COSTE **********************
     
PROJECT  : ST7MC Rx battery charger
COMPILER : ST7 COSMIC

MODULE  :  mtc.c
LIBRARY VERSION  :  1.0

CREATION DATE :    01.2006
AUTHOR :      Florent COSTE	

-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

DESCRIPTION : MTC peripheral control routines
              
-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

 ******************************************************************************
 THE SOFTWARE INCLUDED IN THIS FILE IS FOR GUIDANCE ONLY. THE AUTHOR 
 SHALL NOT BE HELD LIABLE FOR ANY DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES
 WITH RESPECT TO ANY CLAIMS ARISING FROM USE OF THIS SOFTWARE.
 ******************************************************************************

******************************************************************************/  

#include "lib.h"			/* General purpose typedefs and macros */
#include "ports.h"
#include "ST7MC_hr.h"
#include "mtc_hr.h"
#include "Timer.h"  
#include "adc.h"
#include "regul.h"
#include "mtc.h"			/* Public motor control function prototypes */
#include "misc.h" 
#include "opamp.h" 
#include "MTC_Settings_Sensorless.h"

                     
/* -------------------------- Private constants ---------------------------- */

			

/* -------------------------- Public variables ---------------------------- */


/* -------------------------- Private variables ---------------------------- */
 


/* -------------------------- Private functions ---------------------------- */
static void RefreshBootstrap(void);
static void AlignRotor(void);
static void MTC_ResetPeripheral(void);

/* -------------------------- Private macros ---------------------------- */


// Below macros allow to validate PWM outputs or put them in their RESET state
#define MTC_EnableOutputs()		( MCRA |= (1<<MOE) )
#define MTC_DisableOutputs()	( MCRA &= ((1<<MOE)^255) )

// Below macros allow to enable/disable the motor control peripheral clock
#define MTC_EnableClock()		( MCRA |= (1<<CKE) )
#define MTC_DisableClock()		( MCRA &= ((1<<CKE)^255) )

// Below macros allow to enable/disable Direct Access mode to PWM outputs
#define MTC_EnableDirectAccess()		( MCRA |= (1<<DAC) )
#define MTC_DisableDirectAccess()		( MCRA &= ((1<<DAC)^255) )


#define OUTPUT_PARITY		0x2A		// b7=b6=0 for BLDC mode (either sensored or sensorless)
													// b1-b3-b5:T2-T4-T6 are ODD (low)switch
													// b0s-b2-b4:T1-T3-T5 are EVEN (high) switch


/* ------------------------------------------------------------------------- */
/* ------------------------------------------------------------------------- */
/* ------------------------ Functions description -------------------------- */
/* ------------------------------------------------------------------------- */
/* ------------------------------------------------------------------------- */


/*-----------------------------------------------------------------------------
ROUTINE Name : MTC_ResetPeripheral

Description: 	Reset all hardware registers related to the motor control 
				peripheral by toggling the RST bit of the MCRB register.
Input/Output:	None
-----------------------------------------------------------------------------*/
void MTC_ResetPeripheral(void)
{
	SetBit(MCFR,RST); 
	ClrBit(MCFR,RST);
}



/*-----------------------------------------------------------------------------
ROUTINE Name : MTC_InitPeripheral

Description:	This function (re)initializes the motor control peripheral from
				any	state (MCU reset, application level initialization, ...).
Input/Output:	None
Comments: 		It must be noticed that part of MPOL and MDTG registers are 
				write once, meaning they cannot be modified anymore once the 
				MTC_InitPeripheral function has been executed.
-----------------------------------------------------------------------------*/
void MTC_InitPeripheral(void)
{
MTC_ResetPeripheral();

// Initialize registers in page 1 first
SET_MTC_PAGE(1);
MCONF = mem_MCONF;
MPWME = (u8)(mem_MPWME);

MPOL = ALL_ACTIVE_HIGH;				// Switch driver input polarity (L6386D)
							// ZVD bit=0; Z and D have opposite edge
							// REO bit=0;read Z event on even channels

MDTG = mem_MDTG; //reset PCN and write the delay value
MDTG = (u8)(mem_MDTG&(0x40));  // disable the complementary PWM mode
	
MPAR = OUTPUT_PARITY;				// Define Odd/Even MCOx outputs

MZFR = mem_MZFR;
MSCR = mem_MSCR;

// Initialize registers in page 0
SET_MTC_PAGE(0);	

MPCR = mem_MPCR;

MCP0L = mem_MCPOL; 
MCP0H = mem_MCPOH;  // 15.625 Khz PWM frequency
	
MCRA = mem_MCRA;

ALL_PWM_OFF;

MCPWL = 0;    	// 0% duty cycle on W Channel 
MCPWH = 0;   
  
MCRC = mem_MCRC;

MCFR = mem_MCFR;
MDFR = mem_MDFR;

MREF &= (u8)(~CFAV_MASK);	// No Chopper, OAZ input

MTC_EnableClock();

MTC_EnableDirectAccess();
MTC_EnableOutputs();

}


/*-----------------------------------------------------------------------------
ROUTINE Name : MTC_StartMotor

Description:	This function performs three actions:
		* Initializes HW registers and SW variables needed in real time	for motor drive
		* prepares the start-up by refreshing the bootstrap capacitors of the high side
		  switch drivers and aligning the rotor in a known position
		* start-up the peripheral circuitry to get expected interrupts
Input/Output:	None
-----------------------------------------------------------------------------*/
void MTC_Charge(void)
{
SET_MUX_Charge;
PWM_T1_ON;
}

/*-----------------------------------------------------------------------------
ROUTINE Name : MTC_StartMotor

Description:	This function performs three actions:
		* Initializes HW registers and SW variables needed in real time	for motor drive
		* prepares the start-up by refreshing the bootstrap capacitors of the high side
		  switch drivers and aligning the rotor in a known position
		* start-up the peripheral circuitry to get expected interrupts
Input/Output:	None
-----------------------------------------------------------------------------*/
void MTC_Discharge(void)
{
SET_MUX_Discharge;
PWM_T2_ON;
}


/*-----------------------------------------------------------------------------
ROUTINE Name : MTC_Stop_Charge_Discharge

Description:	This function disables all motor control related interrupts 
				and switch off all transistors. This let the windings in 
				floating state once they are completely demagnetized.
Input/Output:	None
-----------------------------------------------------------------------------*/
void MTC_Stop_Charge_Discharge(void)
{
//Set_Duty_for_Current_Ref(0);

ALL_PWM_OFF;
}


/*-----------------------------------------------------------------------------
ROUTINE Name : MTC_U_CL_SO_IT

Description:	This is the PWM Update interrupt service routine.
Comments:	U event taken into account on No action done for BLDC motor drive, 
            but flags are reset in case the routine is entered.
-----------------------------------------------------------------------------*/
@interrupt void MTC_U_CL_SO_IT(void)
{  
	if (MISR & PUI_MSK)
		{
		MISR = ((u8)~(PUI_MSK));
		}
	if (MISR & CLI_MSK) 
		{
		MIMR &= ((u8)~(CLIM_MSK));  // disable CL int
		MISR = ((u8)~(CLI_MSK));
//	        SetBit(Power_Motor_Status,OverCurrent);
//		timer_CL_10ms = 30;  // 300 ms temporisation 
		}
	
	SET_MTC_PAGE(1);
	if (MCONF & SOI_MSK) 
		{
		MCONF &= ((u8)~(SOI_MSK)); 
		}
	SET_MTC_PAGE(0);
}




#if defined ( __HIWARE__)		/* test for HIWARE Compiler */
#pragma TRAP_PROC SAVE_REGS	/* additional registers will be saved */
#elif defined ( __CSMC__ )			/* test for Cosmic Compiler */
@interrupt			/* Cosmic interrupt handling */
#endif
/*-----------------------------------------------------------------------------
ROUTINE Name : MCES_SE_IT

Description:	Emergency Stop and Speed Error interrupt service routine.
        	As this routine is only entered in case of HW problem, the 
	        motor is stopped and trace of this interrupt is keeped with a 
		MotorStatus flag.

Comments:	The Speed Error Flag is only used for induction motor drive.
-----------------------------------------------------------------------------*/
void MCES_SE_IT(void)
{
	MISR = ((u8)~EI_MSK);	// Clr Emergency Stop Flag
}  


/*-----------------------------------------------------------------------------
ROUTINE Name : Set_Duty

Description:	This is the update routine of the PWM duty cycle -> feed MCPUH/L
		(voltage mode) or MCPVH/L (current mode) registers. 
		
Comments:	None
-----------------------------------------------------------------------------*/
void Set_Duty_for_Current_Ref(u16 duty)
{
	ToCMPxL(MCPWL,(u16)duty);
	ToCMPxH(MCPWH,(u16)duty);
}      

/*-----------------------------------------------------------------------------
ROUTINE Name : Set_Duty_for_Voltage_Mode

Description:	This is the update routine of the PWM duty cycle -> feed MCPUH/L
		(voltage mode) or MCPVH/L (current mode) registers. 
		
Comments:	None
-----------------------------------------------------------------------------*/
void Set_Duty_for_Voltage_Mode(u16 duty)
{
	ToCMPxL(MCPUL,(u16)duty);
	ToCMPxH(MCPUH,(u16)duty);
}      


/*** (c) 2006 ****************** END OF FILE ***/
