/**************** (c) 2006 Florent COSTE **********************
     
PROJECT  : ST7MC Rx battery charger
COMPILER : ST7 COSMIC

MODULE  :  i2c.c
LIBRARY VERSION  :  1.0

CREATION DATE :    01.2006
AUTHOR :      Florent COSTE	

-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

DESCRIPTION :   I2C routines 
              
-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

 ******************************************************************************
 THE SOFTWARE INCLUDED IN THIS FILE IS FOR GUIDANCE ONLY. THE AUTHOR 
 SHALL NOT BE HELD LIABLE FOR ANY DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES
 WITH RESPECT TO ANY CLAIMS ARISING FROM USE OF THIS SOFTWARE.
 ******************************************************************************

******************************************************************************/  


#include "lib.h"
#include "ST7MC_hr.h"
#include "i2c.h"
#include "ports.h"
#include "opamp.h"
#include "timer.h"
#include "it_ST7MC.h"  
#include "misc.h"
#include "wwdg.h" 
#include "regul.h"
#include "adc.h"
#include "MTC_Settings_Sensorless.h"
#include "mtc.h"  
#include "lcd.h"  

/* STATIC DECLARATIONS *******************************************************/


struct Model_Struct Model;

/* PUBLIC DECLARATIONS *******************************************************/
static unsigned char ii2c;
static unsigned char sub_idx;

u8 I2c_EepBuff [I2C_EEPROM_SIZE]; 

/*-----------------------------------------------------------------------------
ROUTINE NAME : I2Cm_Start
INPUT/OUTPUT : None.
DESCRIPTION  : Generates I2C-Bus Start Condition.
COMMENTS     :
-----------------------------------------------------------------------------*/ 
static void I2Cm_Start (void)
{
   #asm
      bset PxDR_SDA,#SDA ; SDA_HI
      nop
      nop
      nop
      bset PxDR_SCL,#SCL ; SCL_HI
      nop
      nop
      nop
      bres PxDR_SDA,#SDA ; SDA_LO
      nop
      nop
      nop
      bres PxDR_SCL,#SCL ; SCL_LO
	#endasm
} 

/*-----------------------------------------------------------------------------
ROUTINE NAME : I2Cm_Stop
INPUT/OUTPUT : None.
DESCRIPTION  : Generates I2C-Bus Stop Condition.
COMMENTS     :
-----------------------------------------------------------------------------*/ 
static void I2Cm_Stop (void)
{
    #asm
    bres  PxDR_SDA,#SDA ; SDA_LO
    nop
    nop
    nop
    bset  PxDR_SCL,#SCL ; SCL_HI
    nop
    nop
    nop
    bset  PxDR_SDA,#SDA ; SDA_HI
	#endasm
} 

/*-----------------------------------------------------------------------------
ROUTINE NAME : I2Cm_Ack
INPUT/OUTPUT : None.
DESCRIPTION  : Acknoledge generation from now.
COMMENTS     : Transfer sequence = DATA, ACK.
-----------------------------------------------------------------------------*/ 
static void I2Cm_Ack (void)
{
    #asm
       bset  PxDDR_SDA,#SDA   ; configure  SDA in  OUT
       nop
       nop
       nop
       nop
       nop
       nop
       bres  PxDR_SDA,#SDA   ; SDA_LO  (SEND ACK)
       nop
       nop
       nop
       nop
       nop
       nop
       bset  PxDR_SCL,#SCL   ; SCL_HI
       nop
       nop
       nop
       nop
       nop
       nop
       nop
       nop
       nop
       nop
       nop
       bres  PxDR_SCL,#SCL   ; SCL_LO
       nop
       nop
       nop
       nop
       nop
       nop
       bset  PxDR_SDA,#SDA   ; SDA release (HI)
       nop
       nop
       nop
       nop
       nop
       nop
 	#endasm
} 

/*-----------------------------------------------------------------------------
ROUTINE NAME : I2Cm_nAck
INPUT/OUTPUT : None.
DESCRIPTION  : Non acknoledge generation from now.
COMMENTS     : Transfer sequence = DATA, NACK.
-----------------------------------------------------------------------------*/ 
static void I2Cm_nAck (void)
{
    #asm
    bset  PxDR_SDA,#SDA   
    nop
    nop
    nop
    nop
    nop
    nop
    nop
    nop
    bset  PxDR_SCL,#SCL   ; SCL_HI
    nop
    nop
    nop
    nop
    nop
    nop
    nop
    nop
    nop
    bres  PxDR_SCL,#SCL   ; SCL_LO
    bset  PxDDR_SDA,#SDA   ; configure  SDA in OUT
    nop
    nop
    nop
    nop
    nop
    nop
    nop
    nop
	#endasm
} 

/*-----------------------------------------------------------------------------
ROUTINE NAME : I2Cm_TxData
INPUT/OUTPUT : data byte to be transfered / None.
DESCRIPTION  : Transmits a data byte.
COMMENTS     : Transfer sequence = DATA, ACK, EV8...
-----------------------------------------------------------------------------*/ 
static void I2Cm_TxData (char i2c_data)
{
 #asm
        push  x
        ld  x,#8    ; 8bits
   Tx8_1: sll a
        jrc BitHi1
 BitLo1: bres  PxDR_SDA,#SDA ; SDA<--0
        jp  BitEnd1
 BitHi1: bset  PxDR_SDA,#SDA ; SDA<--1
BitEnd1:
        nop
        nop
        nop
        bset  PxDR_SCL,#SCL ; SCL_HI
        nop
        nop
        nop
        bres  PxDR_SCL,#SCL ; SCL_LO
        nop
        nop
        nop
        dec   x
        jrne  Tx8_1
ackn1:
        bset  PxDR_SDA,#SDA
        nop
        nop
        nop
        bset  PxDR_SCL,#SCL ; SCL_HI
        nop
        nop
        nop
        bres  PxDR_SCL,#SCL ; SCL_LO
        rcf     ; CAK !
        pop x
   #endasm
} 

/*-----------------------------------------------------------------------------
ROUTINE NAME : I2Cm_SetAddr
INPUT/OUTPUT : external I2C device address / None.
DESCRIPTION  : Generates Start-bit and transmits the address byte.
COMMENTS     : Transfer sequence = START, EV5, ADD, ACK...
-----------------------------------------------------------------------------*/ 
void I2Cm_SetAddr (char i2c_addr)
{
    I2Cm_TxData (i2c_addr); 
} 

/*-----------------------------------------------------------------------------
ROUTINE NAME : I2Cm_RxData
INPUT/OUTPUT : Last byte to receive flag (active high) / Received data byte.
DESCRIPTION  : Receive a data byte.
COMMENTS     : Transfer sequence = DATA, ACK, EV7...
-----------------------------------------------------------------------------*/ 
static char I2Cm_RxData (void)
{
      #asm
       push  x
       clr   a
       ld    x,#8
       bres  PxDDR_SDA,#SDA  ; SDA (input direction)
       bset  PxDR_SDA,#SDA   ; SDA_HI( to read)
	nop
	nop
	nop
	nop
        nop
        nop
        nop
        nop
        nop
        nop
Rx_8bits1:
       rlc   a     ; A << SDA  set the LSB
       and   a,#0xFE
       bset  PxDR_SCL,#SCL  ; SCL_HI
       nop
       nop
       nop
       nop
        nop
        nop
        nop
        nop
        nop
        nop
        nop
        nop
       btjf  PxDR_SDA,#SDA,R_lo1        ;if SDA=0 reset
       or    a,#0x01
R_lo1:  bres  PxDR_SCL,#SCL   ; SCL_LO
       dec   x
       nop
       nop
       nop
       nop
        nop
        nop
        nop
        nop
        nop
        nop
        nop
        nop
       jrne  Rx_8bits1     
       rcf         ;reset carry flag
;       bset  PxDDR_SDA,#SDA  ; SDA (out direction)
       
;A=DATA RECEIVED
       pop   x
   #endasm
}


/*-----------------------------------------------------------------------------
ROUTINE NAME : Wait_I2C   
INPUT/OUTPUT : 
DESCRIPTION  : Receive in data buffer via I2C.
COMMENTS     : Low significant bytes first.
-----------------------------------------------------------------------------*/ 
static void Wait_I2C(void)
{
unsigned char cnt;

for(cnt=0;cnt<=10;cnt++)
{};
}



/*-----------------------------------------------------------------------------
ROUTINE NAME : I2Cm_Tx
INPUT/OUTPUT : address, modo_i2c, subaddress, Nb data bytes to transmit/ None.
DESCRIPTION  : Transmit data buffer via I2C.
COMMENTS     : modo_i2c=0 no subaddress --> subaddr=offset from buffer 1st loc
               modo_i2c=1 subaddress + Nb data
               If Nbytes=0 no data, only device address+(subaddress)
-----------------------------------------------------------------------------*/ 
void I2Cm_Tx ( u8 dest_add, u8 sub_add, u8 nb )
{
  I2Cm_Start ();

  I2Cm_SetAddr(dest_add);              /* Slave address selection on I2C bus.*/
  I2Cm_TxData(sub_add);         /* Send Out start location address*/
        
  for (ii2c=0;ii2c<nb; ii2c++)   /* Loop to send all selected data from output buffer.*/
  	I2Cm_TxData(I2c_EepBuff[ii2c]);/* Next output buffer data byte sent.*/
  
  I2Cm_Stop();            /* End of communication: stop condition generation.*/
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : I2Cm_Rx
INPUT/OUTPUT : I2c dest @, address,  Nb databytes to receive / None.
DESCRIPTION  : Read mode for EEPROM like I2C peripherals
COMMENTS     : Low significant bytes first. Fill I2c_EepBuff buffer from adress 0x00 with selected number of datas (nb)
-----------------------------------------------------------------------------*/ 
void I2Cm_Rx (u8 dest_add,u8 offs_add, u8 nb )
{
  ii2c=0;

  I2Cm_Start();
  I2Cm_SetAddr(dest_add);              /* Slave address selection on I2C bus.*/
  I2Cm_TxData(offs_add);                             /* Sub address selection.*/

  I2Cm_Start ();
  I2Cm_SetAddr(dest_add|0x01);       /* Slave address selection in read mode.*/

    
  do                    /* Loop to send all selected data from output buffer.*/
  {
    nb--;
    if (nb==0)                                       /* Last byte to receive.*/
    {
      I2c_EepBuff [ii2c] = I2Cm_RxData();  /* Last output buffer data byte sent.*/
      I2Cm_nAck();                    /* Non acknoledge after last reception.*/
    }
    else
    {
      I2c_EepBuff [ii2c] = I2Cm_RxData();  /* Next output buffer data byte sent.*/
      I2Cm_Ack();                                  /* Acknoledge after reception.*/
    }
    ii2c++;
  } while (nb > 0);          /* Loop to receive the selected number of bytes.*/
  I2Cm_Stop();  /* End of communication: stop condition generation.*/

}




/*-----------------------------------------------------------------------------
ROUTINE NAME : I2C_EepromLoad
INPUT/OUTPUT : 
DESCRIPTION  : Load settings stored in Eeprom 
COMMENTS     : 
-----------------------------------------------------------------------------*/ 
void I2C_EepromLoad(void)            
{
u8 i;

I2c_EepBuff[0] = CRC1_Value;   
I2c_EepBuff[1] = CRC2_Value;   
//I2c_EepBuff[2] = CRC3_Value;   
//I2c_EepBuff[3] = CRC4_Value;   
I2c_EepBuff[2] = 0xff;	// LiION disabled   
I2c_EepBuff[3] = 0xff;  // LiION disabled    
I2c_EepBuff[4] = 0xff;  // delta peak sensitivity HIGH  
I2c_EepBuff[5] = 0xff;  // delta peak sensitivity HIGH   
I2c_EepBuff[6] = 0xff;  // delta peak sensitivity HIGH   
I2c_EepBuff[7] = 0xff;  // delta peak sensitivity HIGH

I2Cm_Tx(I2C_EEP_ADR,0,8);
Wait(5);  // 10 ms delay

for (i=0;i<=6;i++)	Model.Model_Name[i] = disp_message[80+i];
Model.Model_Name[7] = '1';   // model #1 by default

Model.I_Discharge_Start = 800;
Model.I_Discharge_Stop = 300;
Model.I_Charge = 500;
Model.Timeout = 0x011e;  // 0x01 for 1 hour 0x1e for 30 minutes
Model.Sensitivity = HIGH;
Model.VCutoff = 40;  // 4.0V
Model.LiPO = FALSE;	 // NiMh/NiCD by default
Model.LiION = FALSE;	 // NiMh/NiCD by default

for (i=0;i<=TOTAL_MODELS-1;i++)
	{
	u8 j;
	
	for (j=0;j<=7;j++)	I2c_EepBuff[j] = disp_message[80+j];
	I2c_EepBuff[6] = '1' + i;   // model #x  (x=1..2...14)
	if (i>=9) 
		{
		I2c_EepBuff[6] = '1';
		I2c_EepBuff[7] = '0' + (i-9);
		}
	I2Cm_Tx(I2C_EEP_ADR,(u8)((Model_BYTE_SIZE*i) + EEPROM_MODEL_1_BASE_ADRESS),8);  // store model name
	Wait(5);  // 10 ms delay

	I2c_EepBuff[0] = (u8)(Model.I_Discharge_Start>>8);   
	I2c_EepBuff[1] = (u8)(Model.I_Discharge_Start&0xff);   
	I2c_EepBuff[2] = (u8)(Model.I_Discharge_Stop>>8);   
	I2c_EepBuff[3] = (u8)(Model.I_Discharge_Stop&0xff);   
	I2c_EepBuff[4] = (u8)(Model.I_Charge>>8);   
	I2c_EepBuff[5] = (u8)(Model.I_Charge&0xff);   
	I2c_EepBuff[6] = (u8)(Model.Timeout>>8);   
	I2c_EepBuff[7] = (u8)(Model.Timeout&0xff);   

	I2Cm_Tx(I2C_EEP_ADR,(u8)((Model_BYTE_SIZE*i) + EEPROM_MODEL_1_BASE_ADRESS + 8),8);  // store datas
	Wait(5);  // 10 ms delay
	}

	I2c_EepBuff[0] = (u8)0xff;
	I2c_EepBuff[1] = (u8)0xff;
	I2c_EepBuff[2] = (u8)0; // last model number before shutdown
	I2Cm_Tx(I2C_EEP_ADR,(u8)((Model_BYTE_SIZE*TOTAL_MODELS) + EEPROM_MODEL_1_BASE_ADRESS),3);  // store datas
	Wait(5);  // 10 ms delay

	{
	u8 j;
	for (j=0;j<=7;j++)	I2c_EepBuff[j] = (u8)(Model.VCutoff);  // 4.0V, NiMH/NiCD
	}
	I2Cm_Tx(I2C_EEP_ADR,(u8)(EEPROM_MODEL_LiPO_VCUTOFF_BASE_ADRESS),8);  // store 8 datas
	Wait(5);  // 10 ms delay
	I2Cm_Tx(I2C_EEP_ADR,(u8)(EEPROM_MODEL_LiPO_VCUTOFF_BASE_ADRESS + 8),8);  // store 8 datas
	Wait(5);  // 10 ms delay
}


void I2C_EepromRead(u8 offset, u8 data_number)            
{
I2Cm_Rx(I2C_EEP_ADR,offset,data_number);  // read EEPROM
}


void Load_Model_Parameters(u8 model_index)
{
u8 i;
u16 tp;

I2C_EepromRead(EEPROM_MODEL_1_BASE_ADRESS + (Model_BYTE_SIZE*model_index),8);  // read model name

for (i=0;i<=7;i++)	Model.Model_Name[i] = I2c_EepBuff[i];

I2C_EepromRead(EEPROM_MODEL_1_BASE_ADRESS + 8 + (Model_BYTE_SIZE*model_index),8);  // read other parameters

Model.I_Discharge_Start = (u16)(I2c_EepBuff[0]<<8) + I2c_EepBuff[1];   
Model.I_Discharge_Stop = (u16)(I2c_EepBuff[2]<<8) + I2c_EepBuff[3];   
Model.I_Charge = (u16)(I2c_EepBuff[4]<<8) + I2c_EepBuff[5];
Model.Timeout = (u16)(I2c_EepBuff[6]<<8) + I2c_EepBuff[7];


I2C_EepromRead((u8)(EEPROM_LIION_SELECT_BASE_ADRESS),2);  // read 4 bytes

if (model_index<=7)
	{
	u8 tp;
	
	tp = I2c_EepBuff[0] >> (model_index);
	tp &= 0x01;
	
	if (tp == 0) Model.LiION = TRUE;
	else Model.LiION = FALSE;
	}
else
	{
	u8 tp;
	
	tp = I2c_EepBuff[1] >> (model_index-8);
	tp &= 0x01;
	
	if (tp == 0) Model.LiION = TRUE;
	else Model.LiION = FALSE;
	}


I2C_EepromRead((u8)(EEPROM_SENSITIVITY_BASE_ADRESS),4);  // read 4 bytes

if (model_index<=3)
	{
	u8 tp;
	
	tp = I2c_EepBuff[0] >> (2*model_index);
	tp &= 0x03;
	
	Model.Sensitivity = tp;
	}
else if (model_index<=7)
	{
	u8 tp;
	
	tp = I2c_EepBuff[1] >> (2*(model_index-4));
	tp &= 0x03;
	
	Model.Sensitivity = tp;
	}
else if (model_index<=11)
	{
	u8 tp;
	
	tp = I2c_EepBuff[2] >> (2*(model_index-8));
	tp &= 0x03;
	
	Model.Sensitivity = tp;
	}
else
	{
	u8 tp;
	
	tp = I2c_EepBuff[3] >> (2*(model_index-12));
	tp &= 0x03;
	
	Model.Sensitivity = tp;
	}

I2C_EepromRead((u8)(EEPROM_MODEL_LiPO_VCUTOFF_BASE_ADRESS + model_index),1);  // read LiPo & V cutoff  parameters

if ValBit(I2c_EepBuff[0],7) (Model.LiPO = TRUE);
else (Model.LiPO = FALSE);
Model.VCutoff = (u8)(I2c_EepBuff[0]& 0x7f);  // save cutoff value
}


void Save_Model_Parameters(u8 model_index)
{
u8 i;

for (i=0;i<=7;i++)	I2c_EepBuff[i] = Model.Model_Name[i];
I2Cm_Tx(I2C_EEP_ADR,(u8)((Model_BYTE_SIZE*model_index) + EEPROM_MODEL_1_BASE_ADRESS),8);  // store model name
Wait(5);  // 10 ms delay

I2c_EepBuff[0] = (u8)(Model.I_Discharge_Start>>8);   
I2c_EepBuff[1] = (u8)(Model.I_Discharge_Start&0xff);   
I2c_EepBuff[2] = (u8)(Model.I_Discharge_Stop>>8);   
I2c_EepBuff[3] = (u8)(Model.I_Discharge_Stop&0xff);   
I2c_EepBuff[4] = (u8)(Model.I_Charge>>8);   
I2c_EepBuff[5] = (u8)(Model.I_Charge&0xff);   
I2c_EepBuff[6] = (u8)(Model.Timeout>>8);   
I2c_EepBuff[7] = (u8)(Model.Timeout&0xff);   

I2Cm_Tx(I2C_EEP_ADR,(u8)((Model_BYTE_SIZE*model_index) + EEPROM_MODEL_1_BASE_ADRESS + 8),8);  // store datas
Wait(5);  // 10 ms delay


I2C_EepromRead((u8)(EEPROM_LIION_SELECT_BASE_ADRESS),2);  // read 2 bytes

if (model_index<=7)
	{
	I2c_EepBuff[0] &= ~((0x01) << (model_index));
	if (Model.LiION == FALSE)	I2c_EepBuff[0] |= (1 << (model_index));
	}
else
	{
	I2c_EepBuff[1] &= ~((0x01) << (model_index-8));
	if (Model.LiION == FALSE)	I2c_EepBuff[1] |= (1 << (model_index-8));
	}

I2Cm_Tx(I2C_EEP_ADR, EEPROM_LIION_SELECT_BASE_ADRESS, 2);  // store datas
Wait(5);  // 10 ms delay



I2C_EepromRead((u8)(EEPROM_SENSITIVITY_BASE_ADRESS),4);  // read 4 bytes

if (model_index<=3)
	{
	u8 tp;
	
	tp = Model.Sensitivity;
	I2c_EepBuff[0] &= ~(0x03 << (2*model_index));
	I2c_EepBuff[0] |= tp << (2*model_index);
	}
else if (model_index<=7)
	{
	u8 tp;
	
	tp = Model.Sensitivity;
	I2c_EepBuff[1] &= ~(0x03 << (2*(model_index-4)));
	I2c_EepBuff[1] |= tp << (2*(model_index-4));
	}
else if (model_index<=11)
	{
	u8 tp;
	
	tp = Model.Sensitivity;
	I2c_EepBuff[2] &= ~(0x03 << (2*(model_index-8)));
	I2c_EepBuff[2] |= tp << (2*(model_index-8));
	}
else
	{
	u8 tp;
	
	tp = Model.Sensitivity;
	I2c_EepBuff[3] &= ~(0x03 << (2*(model_index-12)));
	I2c_EepBuff[3] |= tp << (2*(model_index-12));
	}

I2Cm_Tx(I2C_EEP_ADR, EEPROM_SENSITIVITY_BASE_ADRESS, 4);  // store datas
Wait(5);  // 10 ms delay

	
I2c_EepBuff[2] = model_index; // model number
I2Cm_Tx(I2C_EEP_ADR,(u8)((Model_BYTE_SIZE*TOTAL_MODELS) + EEPROM_MODEL_1_BASE_ADRESS),3);  // store model number
Wait(5);  // 10 ms delay

I2c_EepBuff[0] = (u8)(Model.VCutoff);  // cutoff value
if (Model.LiPO == TRUE)	SetBit(I2c_EepBuff[0],7);
else ClrBit(I2c_EepBuff[0],7);

I2Cm_Tx(I2C_EEP_ADR,(u8)(EEPROM_MODEL_LiPO_VCUTOFF_BASE_ADRESS + model_index),1);  // store data
Wait(5);  // 10 ms delay
}

/*** (c) 2006  ****************** END OF FILE ***/
