/**************** (c) 2006 Florent COSTE **********************
     
PROJECT  : ST7MC Rx battery charger
COMPILER : ST7 COSMIC

MODULE  :  i2c.c
LIBRARY VERSION  :  1.0

CREATION DATE :    01.2006
AUTHOR :      Florent COSTE	

-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

DESCRIPTION :   ADC routines 
              
-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

 ******************************************************************************
 THE SOFTWARE INCLUDED IN THIS FILE IS FOR GUIDANCE ONLY. THE AUTHOR 
 SHALL NOT BE HELD LIABLE FOR ANY DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES
 WITH RESPECT TO ANY CLAIMS ARISING FROM USE OF THIS SOFTWARE.
 ******************************************************************************

******************************************************************************/  

#include "lib.h"
#include "ST7MC_hr.h"
#include "mtc.h" 
#include "it_st7mc.h" 
#include "lcd.h"
#include "ports.h"
#include "i2c.h"
#include "mtc_hr.h"
#include "adc.h"



u32 Bat_Capacity;
u16 max, Buffer[20], Counter_Capacity_Samples, capa;
u8 event, Flag_1;

SystStatus_Battery Chk_Bat_State;

/*-----------------------------------------------------------------------------
ROUTINE Name : ADC_Get_10bits

Description:	Return the full ADC resolution value (10 bits).
Input/Output:	u8/u16 (channel number/result of conversion)
Comments: 		None
-----------------------------------------------------------------------------*/
u16 ADC_Get_10bits(u8 Channel)		
{
u16 result;
u8 i;

result = 0;	

ADCDRH;	// clear EOC bit in case of...
ADCCSR = Channel;		// 4 Mhz sampling, ADC on

while (!ValBit(ADCCSR,EOC));	// wait till end of conversion
ADCDRH;                       // and ignore 1st result

for (i=0;i<=7;i++)      // take 8 samples
	{
	while (!ValBit(ADCCSR,EOC));	// wait till end of conversion
	result += ADCDRL;
	result += (u16)(ADCDRH << 2);	// clear EOC bit
	}

ClrBit(ADCCSR,ADON);	// shutdown ADC peripheral

result = result >> 3;	// div/8 -> smooth result
return (result);
}
 


/*-----------------------------------------------------------------------------
ROUTINE Name : ADC_Get_8bits

Description:	Return 8 bits resolution ADC value only.
Input/Output:	u8/u8 (channel number/result of conversion)
Comments: 		None
-----------------------------------------------------------------------------*/
u8 ADC_Get_8bits(u8 Channel)	
{
u16 result;
u8 i;

result = 0;	

ADCDRH;	// clear EOC bit in case of...
ADCCSR = Channel;		// 4 Mhz sampling, ADC on

while (!ValBit(ADCCSR,EOC));	// wait till end of conversion
ADCDRH;                       // and ignore 1st result

for (i=0;i<=7;i++)      // take 8 samples
	{
	while (!ValBit(ADCCSR,EOC));	// wait till end of conversion
	result += ADCDRH;	// clear EOC bit
	}

ClrBit(ADCCSR,ADON);	// shutdown ADC peripheral

result = result >> 3;	// div/8 -> smooth result
return ((u8)result);
}
         

/*-----------------------------------------------------------------------------
ROUTINE Name :  Get_Battery_Voltage

Description:	Return Tthe battery voltage in mV

Input/Output:	none/16 bits value
Comments: 		None
-----------------------------------------------------------------------------*/
u16 Get_Battery_Voltage(void)
{
u32 temp;

temp = ADC_Get_10bits(CONVERT_AIN0);
//temp = temp*5*(62+47)*1000*4;
//temp = temp*5*(62+66)*1000*4;	// new value -> 6Kohm
temp = temp*5*(62+82+10)*1000*4;	// new value -> 7.5 Kohm
temp /= 62;
temp /= (1024*4);

if ( (State == DISCHARGE_1) || (State == DISCHARGE_2) ) // need to add voltage drop on 100 mOhm resistor
	{
	temp += (u16)(Get_Current()/10);			
	}
else if ( (State == CHARGE_NIMH) || (State == CHARGE_LIPO_PHASE_1) 
						|| (State == CHARGE_LIPO_PHASE_2) ) // need to subtract voltage drop on 100 mOhm resistor
					{
					if (temp >= Get_Current()/10)	temp -= (u16)(Get_Current()/10);			
					}
return(temp);
}

/*-----------------------------------------------------------------------------
ROUTINE Name :  Get_Current

Description:	Return charging/discharging current in mA

Input/Output:	none/16 bits value
Comments: 		None
-----------------------------------------------------------------------------*/
u16 Get_Current(void)
{
u32 temp;

temp = ADC_Get_10bits(CONVERT_AIN4);
//temp = temp*5*1000*500;
temp = temp*10000*250;
//temp /= (21*50);  // OPAMP gain * resistor value in moHm
//temp /= (16);
//temp /= (5);  // resistor value in moHm/2
//temp *= 10;
temp /= (920);  // OPAMP gain * 100
temp /= (512);

if (temp<50) return(0);
else return(temp);

}


/*-----------------------------------------------------------------------------
ROUTINE Name :  Get_Capacity

Description:	Return battery capacity in mA/H

Input/Output:	none/16 bits value
Comments: 		None
-----------------------------------------------------------------------------*/
u16 Get_Capacity(void)
{
u32 temp;

Bat_Capacity += ADC_Get_10bits(CONVERT_AIN4);	
//Counter_Capacity_Samples++;  // increased every 30 sec

temp = Bat_Capacity*5*1000*2;
temp /= (92);  // OPAMP gain * 10
//temp *= 10;
temp /= (10);  // resistor value in moHm/10

temp /= 120;   // 120 * 30 sec = 1hour
temp /= 2;

if (temp > 9999) Bat_Capacity = 0;	// restart from 0 mA if more than 10Ah.
return(temp);
}

/*-----------------------------------------------------------------------------
ROUTINE Name :  Chk_Slope

Description:	Return true if delta peak achieved

Input/Output:	none/Boolean
Comments: 		None
-----------------------------------------------------------------------------*/
BOOL Chk_Slope(void)
{
unsigned char i,j;
s8 result;
unsigned int min;

for (i=0;i<=18;i++)
	{
	Buffer[i]=Buffer[i+1];
	}
Buffer[19]=ADC_Get_10bits(CONVERT_AIN0);

if (max<=Buffer[19]) 
	{
	max=Buffer[19];
	event=19;
	}

result=j=0;
min=Buffer[event];

for(i=event;i<=18;i++)
	{
	if (Buffer[i]<max) result++;
	if (Buffer[i+1]>Buffer[i]) result--;
	if (min>Buffer[i+1]) min=Buffer[i+1];
	if (Buffer[i] != Buffer[i+1]) j=1;  // 
	}
	if (event>0) event--;
  if (Model.Sensitivity == HIGH)
		{
		//if ((result>=(s8)10) && (abs(max-min)>=2) && j==1) // original 3 & 6
		if ((result>=(s8)10) && (abs(max-min)>=1) && j==1) // original 3 & 6
			return(TRUE);
		}
  else if (Model.Sensitivity == MEDIUM_1)
		{
		if ((result>=(s8)10) && (abs(max-min)>=4) && j==1) 
			return(TRUE);
		}
  else if (Model.Sensitivity == MEDIUM_2)
		{
		if ((result>=(s8)10) && (abs(max-min)>=6) && j==1) 
			return(TRUE);
		}
	else if ((result>=(s8)10) && (abs(max-min)>=8) && j==1) 
		return(TRUE);

return(FALSE);
}




BOOL Chk_Battery_connection(void)  // return TRUE if battery connected
{
	switch (Chk_Bat_State)
			{
			default:
			case BAT_CHK_WAIT_CHARGE:
				if (Chk_Battery_Timer == 0)	
					{
					SetBit(MCRA,3); // current mode
					Chk_Bat_State = BAT_CHK_CHARGE;
					Set_Duty_for_Current_Ref(0);  // current off
					MTC_Stop_Charge_Discharge();
					Wait(6); // 12 ms
					}
				break;
			
			case BAT_CHK_CHARGE:		// test if Vbatt<8.7V with 100 mA charging current
				if (ADC_Get_10bits(CONVERT_AIN0) >= No_BATT_CHARGE)
					{
					Chk_Battery_Timer = 200; //400 ms tempo
					Set_Duty_for_Current_Ref(60);  // <>200ma discharging current for battery detection
					MTC_Discharge();
					Chk_Bat_State = BAT_CHK_WAIT_DISCHARGE;  // try discharge current
					return FALSE;	// not connected
					}
				else 	// connected
					{
					Set_Duty_for_Current_Ref(0);
					MTC_Stop_Charge_Discharge();
					SetBit(MCRA,3); // current mode

					return TRUE;
					}
					break;
				
			case BAT_CHK_WAIT_DISCHARGE:
				if (Chk_Battery_Timer == 0)  // wait 100 ms
					{
					Chk_Bat_State = BAT_CHK_DISCHARGE;
					Set_Duty_for_Current_Ref(0);  // current off
					MTC_Stop_Charge_Discharge();
					Wait(6); // 12 ms
					}
				break;

			case BAT_CHK_DISCHARGE:  	// test if Vbatt>1V with 100 mA discharging current
				if (ADC_Get_10bits(CONVERT_AIN0) <= No_BATT_DISCHARGE)  // chk bat. voltage
					{
					Chk_Battery_Timer = 200; //400 ms tempo

					ClrBit(MCRA,3); // voltage mode
					Set_Duty_for_Current_Ref(250);  // 
//					Set_Duty_for_Current_Ref(150);  // 
					Set_Duty_for_Voltage_Mode(150);
					MTC_Charge();
					Load_Buffer_LCD(14,13);   // no battery
					Blinking_Flag_line1 = 0x00;	// line 1 blinking
					Blinking_Flag_line2 = 0x00;	// line 2 blinking

					Chk_Bat_State = BAT_CHK_WAIT_CHARGE;  // try discharge current
					return FALSE;
					}
				else 	// connected
					{
					Set_Duty_for_Current_Ref(0);
					MTC_Stop_Charge_Discharge();
					SetBit(MCRA,3); // current mode

					return TRUE;
					}
				break;
			}
return (FALSE);
}


u16 Chk_Temperature(void)
{
return(ADC_Get_10bits(CONVERT_AIN14));
}

BOOL Temperature_Protection(void)
{
if( (Chk_Temperature() > Temperature_Threshold_High) && !ValBit(Flag_1,Temperature_Protec_ON) )	SetBit(Flag_1,Temperature_Protec_ON);
else if( (Chk_Temperature() < Temperature_Threshold_Low) && ValBit(Flag_1,Temperature_Protec_ON) )	ClrBit(Flag_1,Temperature_Protec_ON);

if ValBit(Flag_1,Temperature_Protec_ON) return TRUE;
else return FALSE;
}


/*** (c) 2006 ****************** END OF FILE ***/
